/*
 Copyright 2013 Adobe Systems Incorporated.  All rights reserved.

Purpose-
This file has the implementation of Image Editing functionality in Live Edit view
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global $, liveViewObject, dwObject, liveExtensionsConfigPath, dwExtensionController, DW_CONSTANTS, DWLE_DIMENSIONS, DW_EXTENSION_EVENT*/

function AuxiliaryExtensionObject() {
    'use strict';
    
    this.m_currentSelectedElement = null;
    this.m_editable = true;
    this.elementHudHeight = 0; // this is required to compute the position of image hud
    this.elementHudWidth = 0; // this is requireed to compute the position of image hud
    this.elementHudPosOnTop = true; // editable selectors position with respect to element
    this.m_currentHudHeight = 0; // current Hud Height
    this.m_currentHudWidth = 0; // current Hud width
    this.disableDrag = false; // Disable Dragging when mouse is out of Browser Window
    this.m_isShown = false;
    this.m_hudDragged = false;
    this.isListenerAttached = false; // to check if the listener is attached to the element under edit
    this.setCurrentHudTag("");
}

AuxiliaryExtensionObject.prototype.getExtensionId = function () {
    "use strict";
    console.log("base class does not contain the id");
    return "";
};

// setter function to set if element is editable or not
AuxiliaryExtensionObject.prototype.setEditable = function (isEditable) {
    'use strict';
    this.m_editable = isEditable;
};

AuxiliaryExtensionObject.prototype.getEditable = function () {
    'use strict';
    return this.m_editable;
};

AuxiliaryExtensionObject.prototype.getHudTag = function () {
    'use strict';
    return "";
};

AuxiliaryExtensionObject.prototype.getCurrentHudTag = function () {
    'use strict';
    return this.currenthudTag;
};

AuxiliaryExtensionObject.prototype.setCurrentHudTag = function (tagName) {
    'use strict';
    this.currenthudTag = tagName;
};

AuxiliaryExtensionObject.prototype.hideExtensionElementClick = function () {
    var message = {};
    message.commit = true;
    this.hideAuxHud(message);
};

AuxiliaryExtensionObject.prototype.toggleAuxHud = function (message) {
    "use strict";
    if (this.getHudTag().indexOf(message.selectedTag) !== -1) {
        if (this.getVisibility() === false) {
            this.setCurrentHudTag(message.selectedTag);
            this.setVisibility(true);
            this.m_currentSelectedElement = liveViewObject.getCurrentSelectedElement();
            this.isListenerAttached = true; // when we click on element Aux HUD needs to be committed and hidden
            this.m_currentSelectedElement.addEventListener("mouseup", this.hideExtensionElementClick.bind(this), false);
            this.setEditable(liveViewObject.isElementEditable(this.m_currentSelectedElement));
            var extensionID = this.getExtensionId();
            liveViewObject.positionExtensionById(extensionID, 0, 0, parent.document.width, parent.document.height);
            liveViewObject.showExtensionById(extensionID);
            this.populateContent();
            // Update the hud fields with current elemnt props
            if (message.selectorHudHeight) {
                this.elementHudHeight = message.selectorHudHeight;
            } else {
                this.elementHudHeight = DWLE_DIMENSIONS.ElementHudDefault + "px";
            }
            if (message.selectorHudWidth) {
                this.elementHudWidth = message.selectorHudWidth;
            } else {
                this.elementHudWidth = DWLE_DIMENSIONS.ElementHudDefault + "px";
            }
            if (message.selectorHudPosition !== "undefined") {
                this.elementHudPosOnTop = message.selectorHudPosition;
            } else {
                this.elementHudPosOnTop = true;
            }
            this.updateHudFields(this.m_currentSelectedElement);
            var elemRect = liveViewObject.getElementRect(this.getCurrentElement());
            this.positionHud(elemRect);
        } else if (this.getVisibility() === true) {
            message.commit = true;
            this.hideAuxHud(message);
            this.setVisibility(false);
            this.setCurrentHudTag("");
        }
        this.m_hudDragged = false;
    }
};

AuxiliaryExtensionObject.prototype.shouldHide = function () {
    'use strict';
    return true;
};

AuxiliaryExtensionObject.prototype.hideAuxHud = function (message) {
    'use strict';
    var messageDetails = {type: DW_EXTENSION_EVENT.RELOAD_ESH};
    
    if (message && typeof message.shouldScroll !== "undefined") {
        messageDetails.shouldScroll = message.shouldScroll;
    }
    
    if (this.getVisibility() === true && this.shouldHide()) {
        if (message.commit === true) {
            // commit edits before hiding
            this.commit();
        }
        liveViewObject.hideExtensionById(this.getExtensionId());
        this.setVisibility(false);
        this.setCurrentHudTag("");
        messageDetails.type = DW_EXTENSION_EVENT.AUX_HUD_HIDDEN;
        this.m_hudDragged = false;
        
        if (document.activeElement) {
            // If any element in the Hud as focus we should remove it.
            document.activeElement.blur();
        }
        liveViewObject.clearBrowserUndoRedos();
    }            
    if (!message.keepESHOpen) {
        dwExtensionController.sendMessage(messageDetails);
    }
    if (this.isListenerAttached) {
		// if the listener is attached remove it
        this.m_currentSelectedElement.removeEventListener("mouseup", this.hideExtensionElementClick.bind(this), false);
        this.isListenerAttached = false;
    }
};

AuxiliaryExtensionObject.prototype.fileSave = function () {
    'use strict';
    console.log("AuxiliaryExtensionObject::populateContent**Does not have enough information to Save the file. This method should be overriden by the extension to commit changes and save the file on callback**");
};


AuxiliaryExtensionObject.prototype.fileSaveAs = function () {
    'use strict';
    console.log("AuxiliaryExtensionObject::populateContent**Does not have enough information to Save As the file. This method should be overriden by the extension to commit changes and save the file on callback**");
};

AuxiliaryExtensionObject.prototype.initialize = function () {
    'use strict';
    this.setVisibility(false);
    this.setCurrentHudTag("");
    this.m_hudContainer = document.body;
    dwExtensionController.addListener(DW_EXTENSION_EVENT.FILE_SAVE, this.fileSave, this);
    dwExtensionController.addListener(DW_EXTENSION_EVENT.FILE_SAVE_AS, this.fileSaveAs, this);
    dwExtensionController.addListener(DW_EXTENSION_EVENT.HIDE_AUX_HUD, this.hideAuxHud, this);
    dwExtensionController.addListener(DW_EXTENSION_EVENT.TOGGLE_AUX_HUD, this.toggleAuxHud, this);
    
        //add key down handler to handle escape key
    var keyDownHandler = function (evt) {
        evt = evt || window.event;
        if (evt.keyCode === DW_CONSTANTS.EscapeKeyCode) {
            this.escapeKeyPressed();
        }
    };
    
    window.addEventListener("keydown", keyDownHandler.bind(this), false);
};

AuxiliaryExtensionObject.prototype.getCurrentElement = function () {
    'use strict';
    return this.m_currentSelectedElement;
};
AuxiliaryExtensionObject.prototype.elementDimensionChanged = function () {
    'use strict';
    var messageDetails = {};
    messageDetails.type = DW_EXTENSION_EVENT.ELEMENT_DIMENSION_CHANGED;
    dwExtensionController.sendMessage(messageDetails);
};
AuxiliaryExtensionObject.prototype.escapeKeyPressed = function () {
    'use strict';
    
    var messageDetails = {};
    messageDetails.type = DW_EXTENSION_EVENT.AUX_HUD_ESCAPED;
    messageDetails.keyCode = DW_CONSTANTS.EscapeKeyCode;
    dwExtensionController.sendMessage(messageDetails);
};

AuxiliaryExtensionObject.prototype.setVisibility = function (show) {
    'use strict';
    this.m_isShown = show;
    window.parent.liveViewExtensionsObject.setAuxHudVisibility(this.getCurrentHudTag(), this.getExtensionId(), show);
};

AuxiliaryExtensionObject.prototype.getVisibility = function () {
    'use strict';
    return this.m_isShown;
};

AuxiliaryExtensionObject.prototype.populateContent = function () {
    'use strict';
    console.log("AuxiliaryExtensionObject::populateContent**Does not have enough information to populate any content. This method should be overriden by the extension to generate the content/UI**");
};

AuxiliaryExtensionObject.prototype.commit = function () {
    'use strict';
    console.log("AuxiliaryExtensionObject::Commit**Does not have enough information to commit. This method should be overriden by the extension to commit the changes**");
};

AuxiliaryExtensionObject.prototype.updateHudFields = function () {
    'use strict';
    console.log("AuxiliaryExtensionObject::updateHudFields**Does not have enough information to update HUD. This method should be overriden by the extension to Update its UI**");
};

AuxiliaryExtensionObject.prototype.positionHud = function () {
    'use strict';
    console.log("AuxiliaryExtensionObject::positionHud**Does not have enough information to position HUD. This method should be overriden by the extension to position itself properly**");
};

AuxiliaryExtensionObject.prototype.EnableDragOnElement = function (element) {
    'use strict';
    element.onmousedown = function (event) {
        this.disableDrag = false;
        this.dragTheHud(this.m_hudContentDiv, event);
        window.addEventListener("mouseout", this.mouseOutHandler.bind(this));
    }.bind(this);
};

/**
 * @private dragTheHud
 * This function is used for dragging the Hud around the browser surface
 * whenever mouse goes out of the browser we need to remove mouse move listerners to avoid ghost effect.
 * @Param elementToDrag  element being dragged
 * @Param event : mousedown event 
 */
AuxiliaryExtensionObject.prototype.dragTheHud = function (elementToDrag, event) {
    'use strict';
//    this.logDebugMsg("function begin : AuxHUD : dragTheHud");
    var deltaX = event.clientX - parseInt(elementToDrag.style.left, 10);
    var deltaY = event.clientY - parseInt(elementToDrag.style.top, 10);
    document.body.style.webkitUserSelect = "none";
    var auxHudContext = this;
	/*when mouse move happens keep updating the hud position*/
    var moveHandler = function (e) {
        if (!e) {
            e = window.event;
        }
        if (auxHudContext.disableDrag === true) {
			// this means mouse has gone outside the browser window. So remove the handlers for mousemove Otherwise ghosting effect will be there
            if (elementToDrag.removeEventListener) {
                window.removeEventListener("mouseout", this.mouseOutHandler.bind(this));
                document.removeEventListener("mousemove", moveHandler, true);
                document.body.style.webkitUserSelect = "all";
            }
            return;
        }
		// if the new position is inside the browser window then update the position of the HUD
        var newPosX = e.clientX - deltaX;
        var newPosY = e.clientY - deltaY;
        var rightSide = window.innerWidth + window.scrollX - auxHudContext.m_currentHudWidth - DWLE_DIMENSIONS.AuxHudOffSet;
        var bottomSide = window.innerHeight + window.scrollY - auxHudContext.m_currentHudHeight - DWLE_DIMENSIONS.AuxHudOffSet;
        if (newPosX > DWLE_DIMENSIONS.AuxHudOffSet && newPosX < rightSide && newPosY > DWLE_DIMENSIONS.AuxHudOffSet && newPosY < bottomSide) {
            elementToDrag.style.left = newPosX + "px";
            elementToDrag.style.top = newPosY + "px";
        }
        if (e.stopPropagation) {
            e.stopPropagation();
        } else {
            e.cancelBubble = true;
        }
    };
	/*when mouse up happens dragging should stop*/
    var upHandler = function (e) {
        if (!e) {
            e = window.event;
        }
        if (elementToDrag.removeEventListener) {
            document.removeEventListener("mousemove", moveHandler, true);
            document.body.style.webkitUserSelect = "all";
            document.removeEventListener("mouseup", upHandler, true);
            window.removeEventListener("mouseout", auxHudContext.mouseOutHandler.bind(this));
        }
        auxHudContext.m_hudDragged = true;
    };

    if (document.addEventListener) {
        document.addEventListener("mousemove", moveHandler, true);
        document.addEventListener("mouseup", upHandler, true);
    }

};
    
AuxiliaryExtensionObject.prototype.mouseOutHandler = function (e) {
    'use strict';
    this.disableDragging(e);
};

/**
 * @private disableDragging
 * Whenever mouse leaves browser window Disable Dragging of the Image HUD
 * @Param event     : on mouse out comes sometimes comes when we are inside the browser also
 * So need to screen the occurrences.
 */
AuxiliaryExtensionObject.prototype.disableDragging = function (event) {
    'use strict';
//    this.logDebugMsg("function begin : AuxHUD : disableDragging");
    event = event || window.event;
    var toElement = event.relatedTarget || event.toElement;
    if (!toElement) {
        this.disableDrag = true;
    }
};
